<?php

class appGlobKurierBackendActions extends autoAppGlobKurierBackendActions
{
    public function executeAjaxGetPoints()
    {
        if(!$this->getRequest()->isXmlHttpRequest())
        {
            return sfView::NONE;
        }

        $location = stXssSafe::cleanHtml($this->getRequestParameter('location'), 'UTF-8');

        if ($points = GlobKurierPlugin::getGlobBoxPoints($location))
        {
            $responseContent = json_encode($points);
        }
        else
        {
            $this->getResponse()->setStatusCode('404');

            $responseContent = GlobKurierPlugin::$error;
        }

        return $this->renderText($responseContent);
    }

    /*
     * Finds users' data by specific query.
     */
    public function executeAjaxSearchUserData()
    {
        if(!$this->getRequest()->isXmlHttpRequest()) return sfView::NONE;

        $query = trim($this->getRequestParameter('query'));

        $c = new Criteria();

        $c->addJoin(UserDataPeer::COUNTRIES_ID, CountriesPeer::ID, Criteria::LEFT_JOIN);
        $c->addJoin(UserDataPeer::SF_GUARD_USER_ID, sfGuardUserPeer::ID, Criteria::LEFT_JOIN);

        $crit0 = $c->getNewCriterion(UserDataPeer::COMPANY, '%'.$query.'%', Criteria::LIKE);
        $crit1 = $c->getNewCriterion(UserDataPeer::FULL_NAME, '%'.$query.'%', Criteria::LIKE);

        $crit0->addOr($crit1);
        $c->add($crit0);

        $c->addGroupByColumn(UserDataPeer::COUNTRIES_ID);
        $c->addGroupByColumn(UserDataPeer::FULL_NAME);
        $c->addGroupByColumn(UserDataPeer::ADDRESS);
        $c->addGroupByColumn(UserDataPeer::ADDRESS_MORE);
        $c->addGroupByColumn(UserDataPeer::STREET);
        $c->addGroupByColumn(UserDataPeer::HOUSE);
        $c->addGroupByColumn(UserDataPeer::FLAT);
        $c->addGroupByColumn(UserDataPeer::CODE);
        $c->addGroupByColumn(UserDataPeer::TOWN);
        $c->addGroupByColumn(UserDataPeer::PHONE);
        $c->setLimit(20);

        $userDatas = UserDataPeer::doSelect($c);

        $this->users = array();

        foreach ($userDatas as $userData)
        {
            $data = new stdClass();

            $data->is_company = $userData->getCompany() ? true : false;
            $data->is_billing = $userData->getIsBilling() ? true : false;
            $data->name = $data->is_company ? $userData->getCompany() : $userData->getFullName();
            $data->street = $userData->getAddress().' '.$userData->getAddressMore();
            $data->house_number = $userData->getHouse();
            $data->apartment_number = $userData->getFlat();
            $data->postal_code = $userData->getCode();
            $data->city = $userData->getTown();
            $data->country = is_object($userData->getCountries()) ? $userData->getCountries()->getIsoA2() : 'PL';
            $data->contact_person = $userData->getFullName();
            $data->phone = $userData->getPhone();
            $data->email = $userData->getsfGuardUser()->getUsername();

            $this->users[] = array(
                'is_company' => $data->is_company,
                'is_billing' => $data->is_billing,
                'name' => $data->name,
                'data' => json_encode($data)
            );
        }
    }

    /*
     * Gets basic services based on GlobKurierOrder data.
     */
    public function executeAjaxEvaluateOrder()
    {
        if(!$this->getRequest()->isXmlHttpRequest()) return sfView::NONE;

        $orderData = $this->getRequestParameter('glob_kurier_order');
        $this->config = $this->loadConfigOrCreate();

        $data = array(
            'weight' => $orderData['weight'],
            'length' => $orderData['length'],
            'width' => $orderData['width'],
            'height' => $orderData['height'],
            'client_id' => $this->config->get('client_id'),
//            'export' => 'PL',
            'count' => $orderData['number_of_parcels'],
            'delivery' => $orderData['receiver_postal_code']
        );
        if ($orderData['receiver_country'] !== 'PL')
        {
            $data['export'] = $orderData['receiver_country'];
        }
		
        if ($orderData['sender_country'] !== 'PL')
        {
            $data['import'] = $orderData['sender_country'];
        }

        $this->evaluation = GlobKurierApi::call(GlobKurierApi::URL_EVALUATE, $data);
    }

    /*
     * Gets basic addons based on GlobKurierOrder data.
     */
    public function executeAjaxGetOrderAddons()
    {
        if(!$this->getRequest()->isXmlHttpRequest()) return sfView::NONE;

        $this->config = $this->loadConfigOrCreate();

        $data = array(
            'symbol' => $this->getRequestParameter('symbol'),
            'client_id' => $this->config->get('client_id')
        );

        $this->addons = GlobKurierApi::call(GlobKurierApi::URL_GET_ADDONS, $data);
    }

    /**
     * Registers and validates new API user.
     *
     * @return boolean has error
     */
    public function validateConfig()
    {
        $request = $this->getRequest();

        if ($request->getMethod() == sfRequest::POST)
        {
            $i18n = sfI18N::getInstance();

            $this->config = $this->loadConfigOrCreate();

            $data = array_slice(array_change_key_case($request->getParameter('config'), CASE_UPPER), 0, 3);

            $result = GlobKurierApi::call(GlobKurierApi::URL_LOGIN, $data);

            if ($result['status'] != 1)
            {
                $this->setFlash('warning', $result['error']);

                return false;
            }
            else
            {
                $this->config->set('client_id', $result['userParams']['numer_klienta']);

                $this->setFlash('notice', $i18n->__('Wprowadzone dane sa poprawne.'));

                return true;
            }
        }

        return true;
    }

    /**
     * Handles config error.
     */
    public function handleErrorConfig()
    {
        $this->preExecute();
        $this->forward_parameters = $this->getUser()->getAttributeHolder()->getAll('sf_admin/autoAppGlobKurierBackend/forward_parameters');
        $this->config = $this->loadConfigOrCreate();
        $this->labels = $this->getConfigLabels();
        $this->updateConfigFromRequest();

        return sfView::SUCCESS;
    }

    /**
     * Registers and validates new API user.
     *
     * @return boolean has error
     */
    public function validateRegisterConfig()
    {
        $request = $this->getRequest();

        if ($request->getMethod() == sfRequest::POST)
        {
            $i18n = sfI18N::getInstance();

            $data = array_change_key_case($request->getParameter('config'), CASE_UPPER);

            $result = GlobKurierApi::call(GlobKurierApi::URL_REGISTER, $data);

            if ($result['status'] != 1)
            {
                $this->setFlash('warning', $result['error']);

                return false;
            }
            else
            {
                $this->setFlash('notice', $i18n->__('Na podany adres email został wysłany email aktywacyjny.'));

                return true;
            }
        }
    }

    /**
     * Handles register error.
     */
    public function handleErrorRegisterConfig()
    {
        $this->preExecute();
        $this->forward_parameters = $this->getUser()->getAttributeHolder()->getAll('sf_admin/autoAppGlobKurierBackend/forward_parameters');
        $this->config = $this->loadConfigOrCreate();
        $this->labels = $this->getRegisterConfigLabels();
        $this->updateRegisterConfigFromRequest();

        return sfView::SUCCESS;
    }

    /**
     * Shows create (edit) action and sets order reference.
     */
    public function executeCreate()
    {
        $ordersIds = $this->getRequestParameter('order[selected]', array($this->getRequestParameter('order[selected]')));

        if ($ordersIds)
        {
            if (is_array($ordersIds))
            {
                $ordersIds = implode('-', $ordersIds);
            }

            $this->getRequest()->setParameter('glob_kurier_order', array('order_has_glob_kurier_orders' => $ordersIds));
        }

        return $this->forward('appGlobKurierBackend', 'edit');
    }

    /**
     * Validates order.
     *
     * @return boolean has error
     */
    public function validateEdit()
    {
        $request = $this->getRequest();

        $this->glob_kurier_order = $this->getGlobKurierOrderOrCreate();

        $this->updateGlobKurierOrderFromRequest();

        if ($request->getPathInfo() === '/globKurier/save' && $request->getMethod() == sfRequest::POST)
        {
            $i18n = sfI18N::getInstance();

            if ($this->glob_kurier_order->getBaseService())
            {
                return $this->order($this->glob_kurier_order);
            }
            else
            {
                $this->setFlash('warning', $i18n->__('Nie została wybrana usługi podstawowej.'));

                return false;
            }
        }

        return true;
    }

    /**
     * Forwards to editable or non-editable form
     * according to current action's type.
     */
    public function executeEdit()
    {
        if ($this->getRequestParameter('id'))
        {
            return $this->forward('appGlobKurierBackend', 'plainEdit');
        }
        else
        {
            $i18n = sfI18N::getInstance();
            $this->forward_parameters = $this->getUser()->getAttributeHolder()->getAll('sf_admin/autoAppGlobKurierBackend/forward_parameters');

            if ($this->getRequest()->getPathInfo() === '/globKurier/save' && $this->getRequest()->getMethod() == sfRequest::POST)
            {
                $this->saveGlobKurierOrder($this->glob_kurier_order);

                if (!$this->hasFlash('notice'))
                {
                    $this->setFlash('notice', $i18n->__('Twoje zmiany zostały zapisane', null, 'stAdminGeneratorPlugin'));
                }

                if ($this->getRequestParameter('save_and_add'))
                {
                    return $this->redirect('appGlobKurierBackend/create');
                }
                else if ($this->getRequestParameter('save_and_list'))
                {
                    return $this->redirect('appGlobKurierBackend/list');
                }
                else
                {
                    return $this->redirect('appGlobKurierBackend/edit?culture='.$this->getRequestParameter('culture', 'pl_PL').''.'&id='.$this->glob_kurier_order->getId());
                }
            }
            else
            {
                $this->labels = $this->getLabels();
            }

            $this->related_object = $this->glob_kurier_order;
        }
    }

    /*
     * Saves order and redirects to list.
     *
     * @param GlobKurierOrder $globKurierOrder GlobKurier's order
     */
    public function saveGlobKurierOrder($globKurierOrder)
    {
        sfLoader::loadHelpers('Helper');
        use_helper('stUrl');
        $i18n = sfI18N::getInstance();

        parent::saveGlobKurierOrder($globKurierOrder);

        $this->saveOrdersDeliveryNumber($globKurierOrder);

        $this->setFlash('notice', $i18n->__('Zamówienie GlobKurier zostało złożone').', '.st_link_to($i18n->__('pobierz etykietę').'.', 'globKurier/downloadLabel?id='.$globKurierOrder->getId()));

        $this->redirect('globKurier/list');
    }

    /**
     * Gets label from API as downloadable content.
     */
    public function executeDownloadLabel()
    {
        $this->config = $this->loadConfigOrCreate();

        $globKurierOrder = GlobKurierOrderPeer::retrieveByPk($this->getRequestParameter('id'));

        $data = array(
            'LOGIN' => $this->config->get('login'),
            'PASSWORD' => $this->config->get('password'),
            'GKNUMBER' => $globKurierOrder->getNrgk()
        );

        $result = GlobKurierApi::call(GlobKurierApi::URL_GET_LABEL, $data);

        if ($result['status'] != 1)
        {
            $this->setFlash('warning', $result['error']);

            return $this->redirect($this->getRequest()->getReferer());
        }
        else
        {
            $this->redirect($result['url']);
        }
    }

    /**
     * Saves waybills numbers in order.
     *
     * @param type $globKurierOrder
     */
    protected function saveOrdersDeliveryNumber($globKurierOrder)
    {
        $c = new Criteria();

        $c->addJoin(OrderHasGlobKurierOrderPeer::ST_ORDER_ID, OrderPeer::ID, Criteria::LEFT_JOIN);
        $c->addJoin(OrderPeer::ORDER_DELIVERY_ID, OrderDeliveryPeer::ID, Criteria::LEFT_JOIN);
        $c->add(OrderHasGlobKurierOrderPeer::APP_GLOB_KURIER_ORDER_ID, $globKurierOrder->getId());

        $orderDeliverys = OrderDeliveryPeer::doSelect($c);

        foreach ($orderDeliverys as $orderDelivery)
        {
            $orderDelivery->setNumber($globKurierOrder->getNrgk());

            $orderDelivery->save();
        }
    }

    /**
     * Creates GlobKurier's order in API.
     *
     * @param GlobKurierOrder $globKurierOrder order
     *
     * @return boolean is order valid
     */
    protected function order(GlobKurierOrder $globKurierOrder)
    {
        $this->config = $this->loadConfigOrCreate();

        $globKurierOrder->setAdditionalServices($this->getRequestParameter('glob_kurier_order_addons'));

        $courier = explode('||', $globKurierOrder->getBaseService());

        $globKurierOrder->setBaseService($courier[0]);
        $globKurierOrder->setCourier($courier[1]);
		
        $xml = new SimpleXMLElement('<globkurier/>');

        $xml->addChild('login', $this->config->get('login'));
        $xml->addChild('password', $this->config->get('password'));
        $xml->addChild('apikey', $this->config->get('apikey'));
        $xml->addChild('order_method', 'SAVE');
        $xml->addChild('order_source', 'SOTESHOP');
        $xml->addChild('orders');

        $xmlOrder = $xml->orders->addChild('order');
		$date = new DateTime($globKurierOrder->getSentDate());
        $xmlOrder->addChild('sent_date', $date->format('Y-m-d H:i:s'));
        $xmlOrder->addChild('base_service', $globKurierOrder->getBaseService());

        if (is_array($globKurierOrder->getAdditionalServices()))
        {
            $xmlOrder->addChild('additional_services');

            foreach ($globKurierOrder->getAdditionalServices() as $addon)
            {
                $xmlOrder->additional_services->addChild('additional_service', $addon);
            }
        }

        $xmlOrder->addChild('cod_amount', $globKurierOrder->getCodAmount());
        $xmlOrder->addChild('cod_account', $globKurierOrder->getCodAccount());
        $xmlOrder->addChild('insurance_amount', $globKurierOrder->getInsuranceAmount());
        $xmlOrder->addChild('declared_value', $globKurierOrder->getDeclaredValue());
        $xmlOrder->addChild('payment_type', $globKurierOrder->getPaymentType());
        $xmlOrder->addChild('content', $globKurierOrder->getContent());
        $xmlOrder->addChild('length', $globKurierOrder->getLength());
        $xmlOrder->addChild('width', $globKurierOrder->getWidth());
        $xmlOrder->addChild('height', $globKurierOrder->getHeight());
        $xmlOrder->addChild('weight', $globKurierOrder->getWeight());
        $xmlOrder->addChild('number_of_parcels', $globKurierOrder->getNumberOfParcels());
        $xmlOrder->addChild('comments', $globKurierOrder->getComments());

        $xmlOrder->addChild('sender');

        $xmlOrder->sender->addChild('name', $globKurierOrder->getSenderName());
        $xmlOrder->sender->addChild('street', $globKurierOrder->getSenderStreet());
        $xmlOrder->sender->addChild('house_number', $globKurierOrder->getSenderHouseNumber());
        $xmlOrder->sender->addChild('apartment_number', $globKurierOrder->getSenderApartmentNumber());
        $xmlOrder->sender->addChild('postal_code', $globKurierOrder->getSenderPostalCode());
        $xmlOrder->sender->addChild('city', $globKurierOrder->getSenderCity());
        $xmlOrder->sender->addChild('country', $globKurierOrder->getSenderCountry());
        $xmlOrder->sender->addChild('contact_person', $globKurierOrder->getSenderContactPerson());
        $xmlOrder->sender->addChild('phone', $globKurierOrder->getSenderPhone());
        $xmlOrder->sender->addChild('email', $globKurierOrder->getSenderEmail());

        $xmlOrder->addChild('receiver');

        $xmlOrder->receiver->addChild('name', $globKurierOrder->getReceiverName());
        $xmlOrder->receiver->addChild('street', $globKurierOrder->getReceiverStreet());
        $xmlOrder->receiver->addChild('house_number', $globKurierOrder->getReceiverHouseNumber());
        $xmlOrder->receiver->addChild('apartment_number', $globKurierOrder->getReceiverApartmentNumber());
        $xmlOrder->receiver->addChild('postal_code', $globKurierOrder->getReceiverPostalCode());
        $xmlOrder->receiver->addChild('city', $globKurierOrder->getReceiverCity());
        $xmlOrder->receiver->addChild('country', $globKurierOrder->getReceiverCountry());
        $xmlOrder->receiver->addChild('contact_person', $globKurierOrder->getReceiverContactPerson());
        $xmlOrder->receiver->addChild('phone', $globKurierOrder->getReceiverPhone());
        $xmlOrder->receiver->addChild('email', $globKurierOrder->getReceiverEmail());

        if ($globKurierOrder->getSenderTerminal() && $globKurierOrder->getReceiverTerminal())
        {
            $senderTerminal = explode('||', $globKurierOrder->getSenderTerminal());
            $xmlOrder->sender->addChild('terminal', $senderTerminal[0]);

            $receiverTerminal = explode('||', $globKurierOrder->getReceiverTerminal());
            $xmlOrder->receiver->addChild('terminal', $receiverTerminal[0]);
        }

        if ($globKurierOrder->getReceiverGlobBox())
        {
            $v = explode('||', $globKurierOrder->getReceiverGlobBox());

            $xmlOrder->addChild('globbox');

            $xmlOrder->globbox->addChild('sender_type', 'osoba');
            $xmlOrder->globbox->addChild('receiver_type', 'partner');
            $xmlOrder->globbox->addChild('receiver_id_point', $v[0]);
        }

        $data = array(
            'DATA' => $xml->asXML()
        );
		
		//echo $xml->asXML();

        $result = GlobKurierApi::call(GlobKurierApi::URL_ORDER, $data);

        if ($result['status'] != 1)
        {
            $this->setFlash('warning', $result['error']);

            return false;
        }
        else
        {
            $globKurierOrder->setNrgk($result['params']['nrgk']);

            return true;
        }
    }

    /**
     * Populates GlobKurierOrder with shop related data.
     *
     * @param GlobKurierOrder $globKurierOrder
     */
    protected function populateGlobKurierOrderFromDefaultValues(GlobKurierOrder $globKurierOrder)
    {
        $this->config = $this->loadConfigOrCreate();

        $globKurierOrder->setSentDate(date('Y-m-d'));
        $globKurierOrder->setLength($this->config->get('default_length'));
        $globKurierOrder->setWidth($this->config->get('default_width'));
        $globKurierOrder->setHeight($this->config->get('default_height'));
        $globKurierOrder->setWeight($this->config->get('default_weight'));

        $shopInfo = stConfig::getInstance('stShopInfoBackend');

        $globKurierOrder->setSenderName($shopInfo->get('company'));
        $globKurierOrder->setSenderStreet($shopInfo->get('street'));
        $globKurierOrder->setSenderHouseNumber($shopInfo->get('house'));
        $globKurierOrder->setSenderApartmentNumber($shopInfo->get('flat'));
        $globKurierOrder->setSenderPostalCode($shopInfo->get('code'));
        $globKurierOrder->setSenderCity($shopInfo->get('town'));
        $globKurierOrder->setSenderCountry('PL');
//        $globKurierOrder->setSenderContactPerson();
        $globKurierOrder->setSenderPhone($shopInfo->get('phone'));
        $globKurierOrder->setSenderEmail($shopInfo->get('email'));		
		
        $globKurierOrder->setSenderTerminal($this->config->get('sender_terminal'));

        $globKurierOrder->setCodAccount($shopInfo->get('bank'));
        $globKurierOrder->setNumberOfParcels(1);
        $globKurierOrder->setSentDate(date('Y-m-d H:i'));
    }

    /**
     * Gets or creates new GlobKurierOrder. If order id is
     * passed then creates new on order basis.
     *
     * @param integer $id GlobKurierOrder's id
     *
     * $return GlobKurierOrder order
     */
    protected function getGlobKurierOrderOrCreate($id = 'id')
    {
        $event = $this->getDispatcher()->notifyUntil(new sfEvent($this, 'autoAppGlobKurierBackendActions.preGetOrCreate'));

        if (!$event->isProcessed())
        {
            if (!$this->getRequestParameter($id))
            {
                $this->glob_kurier_order = new GlobKurierOrder();

                $this->populateGlobKurierOrderFromDefaultValues($this->glob_kurier_order);

                $ordersIds = $this->getRequestParameter('order[selected]', array($this->getRequestParameter('id')));

                if ($ordersIds)
                {
                    $c = new Criteria();

                    $c->add(OrderPeer::ID, $ordersIds, Criteria::IN);
                    $c->addAscendingOrderByColumn(OrderPeer::CREATED_AT);

                    if ($orders = OrderPeer::doSelect($c))
                    {
                        foreach ($orders as $order)
                        {
                            if ($order->countOrderHasGlobKurierOrders())
                            {
                                $this->setFlash('warning', $this->getContext()->getI18N()->__('Na podstawie zamówienia można stworzyć tylko jedną przesyłkę.'));

                                $this->redirect($this->getRequest()->getReferer());
                            }
                        }

                        $this->glob_kurier_order->populateFromOrders($orders);
                    }
                }
            }
            else
            {
                $this->glob_kurier_order = GlobKurierOrderPeer::retrieveByPk($this->getRequestParameter($id));

                $this->forward404Unless($this->glob_kurier_order);
            }
        }

        $this->getDispatcher()->notify(new sfEvent($this, 'autoAppGlobKurierBackendActions.postGetOrCreate', array('modelInstance' => $this->glob_kurier_order)));

        if (method_exists($this->glob_kurier_order, 'setCulture'))
        {
            if ($this->glob_kurier_order->getPrimaryKey())
            {
                $language = $this->getRequestParameter('culture', stLanguage::getOptLanguage());
            }
            else
            {
                $language = stLanguage::getOptLanguage();
            }

            $this->glob_kurier_order->setCulture($language);
        }

        return $this->glob_kurier_order;
    }
}
